/**
* \file: GstreamerVideoSinkImpl.cpp
*
* \version: $Id:$
*
* \release: $Name:$
*
* Actual implementation of AditVideoSink class (see pImpl idiom)
*
* \component: Baidu CarLife
*
* \author: P. Govindaraju / RBEB/GM / Pradeepa.Govindaraju@in.bosch.com
*
* \copyright (c) 2016-2017 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#include <bdcl/BaiduCoreCallbackDealer.h>
#include <bdcl/AditVideoSink.h>
#include <adit_logging.h>
#include <gst/app/gstappsrc.h>
#include "GstreamerVideoSinkImpl.h"

#define _videoAppSrc "bdcl_video_src"
#define _videoPipelineBin "bdcl_video_out"

LOG_IMPORT_CONTEXT(bdcl_gst)

namespace adit { namespace bdcl {

void* GstreamerVideoSinkImpl::sessionContext = nullptr;

GstreamerVideoSinkImpl::GstreamerVideoSinkImpl(IAditVideoSinkCallbacks* inCallbacks, CoreCallbackDealer* inCallbackDealer)
{
    mCallbacks = inCallbacks;
    firstFrameRendered = false;
    mCallbackDealer = inCallbackDealer;
    sessionContext  = inCallbackDealer;
    firstFrameRenderedHandlerId = 0;
    mIsSetThreadParam = false;

    GError* error = nullptr;
    gboolean ret = gst_init_check(NULL, NULL, &error);//todo Need to check necessity of this block.
    if (ret == TRUE)
    {

    }
    else
    {
        if (error != nullptr)
        {
            LOG_ERROR((bdcl_gst, "gst_init_check failed: %s", error->message));
            g_error_free(error);
        }
        else
        {
            LOG_ERROR((bdcl_gst, "gst_init_check failed (unknown error)"));
        }

        LOG_FATAL((bdcl_gst, "unable to register Gstreamer adapters"));
    }
}

GstreamerVideoSinkImpl::~GstreamerVideoSinkImpl()
{
    teardown();
    MessageBusUnref();
}

void GstreamerVideoSinkImpl::setConfigItem(std::string inKey, std::string inValue)
{
    LOGD_DEBUG((bdcl_gst, "setConfigItem: inKey = %s inValue = %s", inKey.c_str(), inValue.c_str()));
    mConfig.set(inKey, inValue);
}

bool GstreamerVideoSinkImpl::initialize()
{
    if (!mConfig.ResultConfig(false)) {
        LOG_ERROR((bdcl_gst, " could not initialize video sink endpoint due to configuration error"));
        return false;
    }

    mCallbackDealer->registerVideoSinkCallbacks(mCallbacks);
    mCallbackDealer->registerVideoSinkBackendCallbacks(this);

    return true;
}

void GstreamerVideoSinkImpl::teardown()
{
    /* notify that display becomes invalid */
    uspi::SharedDataSender::instance().notifySurfaceDataExpiration(sessionContext);

    if (mVideoPipeline.appsrc != nullptr)
    {
        mVideoPipeline.appsrc = nullptr; // is unref'ed by pipeline
    }
    if (mVideoPipeline.gstVideoSink != nullptr)
    {
        if (firstFrameRenderedHandlerId > 0)
        {
            g_signal_handler_disconnect(mVideoPipeline.gstVideoSink, firstFrameRenderedHandlerId);

            firstFrameRenderedHandlerId = 0;
        }

        gst_object_unref(mVideoPipeline.gstVideoSink);
        mVideoPipeline.gstVideoSink = nullptr;
    }
    // stop
    mVideoPipeline.StopPipeline();

}

void GstreamerVideoSinkImpl::configureStream()
{
    initVideoParam.width = mConfig.mWidth;
    initVideoParam.height = mConfig.mHeight;
    initVideoParam.frameRate = mConfig.mFrameRate;
    CCarLifeLib::getInstance()->cmdVideoEncoderInit(&(initVideoParam));
    LOGD_DEBUG((bdcl_gst, "videoEncoderInit sent"));
}

bool GstreamerVideoSinkImpl::playbackStart()
{
    if (!mConfig.ResultConfig(true)) {
        LOG_ERROR((bdcl_gst, " playbackStart failed due to configuration error"));
        return false;
    }

    mVideoPipeline.CreatePipeline(_videoAppSrc, _videoPipelineBin, mConfig.mPipeline);

    GstCaps* caps = gst_caps_new_simple("video/x-h264",
                                        "width", G_TYPE_INT, mConfig.mWidth,
                                        "height", G_TYPE_INT, mConfig.mHeight,
                                        "framerate", GST_TYPE_FRACTION, 0, 1,
                                        NULL); // todo how to take the width

    if (caps != nullptr)
    {
        gst_app_src_set_caps(GST_APP_SRC(mVideoPipeline.appsrc), caps);
        gst_caps_unref(caps);
    }
    else
    {
        LOG_ERROR((bdcl_gst, "could not set video out caps"));
        return false;
    }

    uint64_t maxBytes = 100000;
    g_object_set(G_OBJECT(mVideoPipeline.appsrc),
            "is-live",      TRUE,               // live source
            "do-timestamp", FALSE,              // don't create timestamps
            "block",        FALSE,              // do not block if queue is full
            "min-percent",  0,                  // always empty queue until 0%
            "max-bytes",    (guint64)maxBytes,  // max queue size
            NULL);
    gst_app_src_set_latency(GST_APP_SRC(mVideoPipeline.appsrc), 0, 100000); // TODO performance: configuration? what is appropriate?

    GstIterator* iter = gst_bin_iterate_sinks(GST_BIN(mVideoPipeline.partial));
    if (iter == nullptr)
    {
        LOG_ERROR((bdcl_gst, "Could not get the iterator from GStreamer for the sinks"));
        return false;
    }
    std::string sink_name = "GstApxSink"; //g_object type name of adit gst_apx_sink
    std::string signal_name = "first-videoframe-rendered"; // first frame rendered signal/callback

    if(gst_iterator_next(iter, (void**)&(mVideoPipeline.gstVideoSink)) == GST_ITERATOR_OK)    //Only one sink will be there
    {
        const std::string name = G_OBJECT_TYPE_NAME(G_OBJECT(mVideoPipeline.gstVideoSink));
        if (name.compare(sink_name) == 0)
        {
            //register first frame-buffer rendered signal handler
            firstFrameRenderedHandlerId = g_signal_connect(mVideoPipeline.gstVideoSink, signal_name.c_str(),
                    G_CALLBACK (&GstreamerVideoSinkImpl::firstFrameRenderedHandler), this);

            gst_iterator_free(iter);

            if (firstFrameRenderedHandlerId == 0)
            {
                LOG_ERROR((bdcl_gst, "Could not connect to first-video-frame signal of Sink %s", name.c_str()));
                gst_object_unref(mVideoPipeline.gstVideoSink);
                return false;
            }
        }
        else
        {
            LOG_ERROR((bdcl_gst, "Could not find video sink '%s' to register for first-videoframe-rendered signal", name.c_str()));
            gst_object_unref(mVideoPipeline.gstVideoSink);
            return false;
        }
    }
    else
    {
        LOG_ERROR((bdcl_gst, "GStreamer iterator did not return a valid value as a sink"));
        return false;
    }

    // then set to STATE_PLAYING (is asynchronous)
    auto ret = gst_element_set_state(GST_ELEMENT(mVideoPipeline.pipeline), GST_STATE_PLAYING);
    if (ret == GST_STATE_CHANGE_FAILURE)
    {
        LOG_ERROR((bdcl_gst, "gst_element_set_state GST_STATE_PLAYING failed"));
        return false;
    }else{
        LOG_INFO((bdcl_gst, "gst_element_set_state GST_STATE_PLAYING success"));

    }

    g_object_get(G_OBJECT (mVideoPipeline.gstVideoSink), "wl_display", &(mDisplay), NULL);
    if(!(mDisplay))
    {
        LOG_ERROR((bdcl_gst,"could not retrieve wl_display"));
        return false;
    }

    //Calling Baidu core callback. after calling this function MD will start streaming the video frames immediately.
    CCarLifeLib::getInstance()->cmdVideoEncoderStart();

    return true;

}
void GstreamerVideoSinkImpl::playbackPause()
{
    CCarLifeLib::getInstance()->cmdVideoEncoderPause();
    //todo need to take care of whether we need to destroy the pipeline or not.
}

void GstreamerVideoSinkImpl::onVideoDataAvailable(uint8_t *data, uint32_t len)
{
    (void) data;
    (void) len;

    if (!mIsSetThreadParam) {
        _setThreadPriority();
        mIsSetThreadParam = true;   // setting thread priority only once
    }

    if (mVideoPipeline.appsrc == nullptr)
    {
        LOG_ERROR((bdcl_gst, "appsrc doesnt exist"));
    }
    else
    {
        // gst_buffer_new_and_alloc() could lead to abort if requested memory cant be allocated
        GstBuffer* buffer = gst_buffer_try_new_and_alloc(len);
        if(buffer != NULL)
        {
            if(GST_BUFFER_DATA(buffer) != NULL)
            {
                memcpy(GST_BUFFER_DATA(buffer), data, len);
//                GST_BUFFER_TIMESTAMP(buffer) = inFragment.GetDisplayTime();
            }
            else
            {
                LOG_ERROR((bdcl_gst, "gstbuffer data pointer NULL"));
                gst_buffer_unref(buffer);
                return;
            }
        }
        else
        {
            LOG_ERROR((bdcl_gst, "gst buffer NULL!!"));
            return;
        }

        int ret = gst_app_src_push_buffer(GST_APP_SRC(mVideoPipeline.appsrc), buffer);
        // takes over buffer, no unref required
        if (ret != 0)
        {
            gst_buffer_unref(buffer); // can't be sure if buffer was unref'ed or not
            // min. interval 60 seconds
            LOG_WARN_RATE(60, (bdcl_gst, "gst_app_src_push_buffer failed: %d (%u times)", ret,
                    LOG_RATE_COUNTER));
        }
    }
}

void GstreamerVideoSinkImpl::_setThreadPriority()
{
    if(!mConfig.mDisablePrio)
    {
        /* set thread priority */
        int err = 0;
        int schedPolicy;
        struct sched_param schedParam;

        err = pthread_getschedparam(pthread_self(), &schedPolicy, &schedParam);
        if(err == 0)
        {
            schedParam.sched_priority = mConfig.mThreadPrio;
            err = pthread_setschedparam(pthread_self(), SCHED_FIFO, &schedParam);
            if(err != 0)
            {
                LOG_ERROR((bdcl_gst, "Failed to set video thread priority with error %d", err));
            }
        }
        else
        {
            LOG_ERROR((bdcl_gst, "Failed to get thread priority with error %d", err));
        }
    }
}

void GstreamerVideoSinkImpl::firstFrameRenderedHandler(GstElement* element, GstPad* pad,
        gpointer user_data)
{
    (void)element;
    (void)pad;
    if(user_data != nullptr)
    {
        GstreamerVideoSinkImpl* tmpPipeline = static_cast<GstreamerVideoSinkImpl*>(user_data);

        if(tmpPipeline->firstFrameRendered == false)
        {
            LOGD_DEBUG((bdcl_gst, "Value of sessionContext: %p", sessionContext));
            LOGD_DEBUG((bdcl_gst, "video first frame rendered"));
            uspi::SharedDataSender::instance().transmitSurfaceData(sessionContext, tmpPipeline->mDisplay);

            if (tmpPipeline->mCallbacks != nullptr)
            {
                tmpPipeline->mCallbacks->onFirstFrameRendered();
            }
            else
            {
                LOG_ERROR((bdcl_gst, "GStreamer callbacks are not set"));
            }
            // no more notification for first-frame rendered
            tmpPipeline->firstFrameRendered = true ;
        }
    }
    else
    {
        LOG_ERROR((bdcl_gst, "firstFrameRenderedHandler::user_data could not be transferred"));
    }

}

} } /*namespace adit { namespace bdcl { */




